' Morse Code Practice
' Rev 1.0.0 William M Leue 11/1/2020

option default integer
option base 1

' Constants
const NUM_ALPHA = 26
const ALPHA_LEN = 4
const NUM_NUMBERS = 10
const NUM_LEN = 5
const NUM_PUNCT = 19
const PUNCT_LEN = 6
const NUM_KOCH = 43
const DIT        = 1
const DAH        = 2
const DIT_LEN    = 1
const DAH_LEN    = 3*DIT_LEN

' Published value is 2.4 but this doesn't seem to include inter-char pauses.
' Using 1.2 produces code speeds very close to desired.
' Max speed is limited by overhead in play sound calls; in particular, how fast
' dits can be turned on and off.
const WPM_RATIO  = 1.2

const MIN_FREQ = 300
const MAX_FREQ = 1500
const MIN_SPEED = 5
const MAX_SPEED = 30

const ALPHA_ASC_START = 65
const ALPHA_ASC_END   = 90
const NUM_ASC_START   = 48
const NUM_ASC_END     = 57

const MIN_BATCH = 10
const MAX_BATCH = 255
const RTEXT_NLINES = 39

' Graphics Constants
const NMENUS = 8
const MSPEED = 1
const MFREQ  = 2
const MBSIZE = 3
const MTXT1  = 4
const MTXT2  = 5
const MTXT3  = 6
const MTXT4  = 7
const MKOCH  = 8

const MENUX  = 100
const MENUXV = 450
const MENUY  = 150
const MENUW  = 500
const MENUH  = 40
const MENUI  = 50

const UP = 128
const DOWN = 129
const LEFT = 130
const RIGHT = 131
const ENTER = 13
const QUERY = 63
const LQUERY = 47
const QUIT = 81
const LQUIT = 113
const CKWIDTH = 15
const CKHEIGHT = 30

const FROM_FILE            = 1
const RAND_ALPHA           = 2
const RAND_ALPHA_NUM       = 3
const RAND_ALPHA_NUM_PUNCT = 4
const KOCH_INCREMENTAL     = 5

const KOCH_WORD_LENGTH = 5

' Globals
dim alpha(NUM_ALPHA, ALPHA_LEN)
dim numbers(NUM_NUMBERS, NUM_LEN)
dim punctuation(NUM_PUNCT, PUNCT_LEN)
dim punct_ascii(NUM_PUNCT)
dim koch_order_char$(NUM_KOCH)
dim float wpm_speed = 13.0
dim dit_duration
dim dah_duration
dim elt_ticks
dim char_ticks
dim word_ticks
dim frequency = 1000
dim koch_char_index = 1
dim num_koch_used = 2
dim prev_choice = 0
dim batch_size = 20
dim text_mode = FROM_FILE
dim text_filename$ = ""
dim ckx(7), cky(7)
dim modename$(5)
dim rtext_lines$(RTEXT_NLINES)
dim rtext_lens(RTEXT_NLINES)

' Main Program
open "debug.txt" for output as #1
SetGraphics
ReadMorse
ReadModeNames
LoadPreferences
MakeDurations
ReadRandomText
do
  DrawScreen
loop
end

' Set up graphics mode
sub SetGraphics
  mode 1,8
end sub

' Read morse code values and ASCII for punctuation
sub ReadMorse
  local i, j
  for i = 1 to NUM_ALPHA
    for j = 1 to ALPHA_LEN
      read alpha(i, j)
    next j
  next i
  for i = 1 to NUM_NUMBERS
    for j = 1 to NUM_LEN
      read numbers(i, j)
    next j
  next i
  for i = 1 to NUM_PUNCT
    for j = 1 to PUNCT_LEN
      read punctuation(i, j)
    next j
  next i
  for i = 1 to NUM_PUNCT
    read punct_ascii(i)
  next i
  for i = 1 to NUM_KOCH
    read koch_order_char$(i)
  next i
end sub

' Read mode name strings
sub ReadModenames
  local i
  for i = 1 to 5
    read modename$(i)
  next i
end sub

' Make durations for Morse Code elements,
' based on the current WPM speed.
sub MakeDurations
  local float r, r1
  local max_dit_duration
  r = WPM_RATIO/(1.0*wpm_speed)
  r1 = WPM_RATIO/(8.0)
  dit_duration = int(1000*r + 0.5)
  max_dit_duration = int(1000*r1 + 0.5)
  if dit_duration > max_dit_duration then
    dit_duration = max_dit_duration
  end if
  dah_duration = 3*dit_duration
  elt_pause    = dit_duration
  char_pause   = dah_duration
  word_pause   = 5*dit_duration
  if dit_duration > max_dit_duration then
    dit_duration = max_dit_duration
    dah_duration = 3*dit_duration
  end if
  'print "dit_duration: " + str$(dit_duration)
  'print "dah_duration: " + str$(dah_duration)
  'print "elt_pause:    " + str$(elt_pause)
  'print "char_pause:   " + str$(char_pause)
  'print "word_pause:   " + str$(word_pause)
end sub

' Read the Random Text file
sub ReadRandomText
  local i
  open "RandomText.txt" for input as #3
  for i = 1 to RTEXT_NLINES
    line input #3, rtext_lines$(i)
    rtext_lens(i) = len(rtext_lines$(i))
  next i
  close #3
end sub

' Play a Test String
sub PlayTestString
  EmitMorseString "Now is the time for all good men to come to the aid of their country"
end sub

' Draw Screen for User Inputs
sub DrawScreen
  cls
  DrawMenu
end sub

sub DrawMenu
  local i, y, cmd, choice
  local z$
  
  cls
  text MM.HRES\2, 20, "Morse Code Training Options", "CT", 5
  text MM.HRES\2, 50, "Use UP and DOWN arrow keys to select menu item", "CT"
  text MM.HRES\2, 70, "Use LEFT and RIGHT arrow keys to change number entries", "CT"
  text MM.HRES\2, 90, "Press ENTER when you are satisfied with your choices.", "CT"
  text MM.HRES\2, 110, "Press '?' for a brief help screen, or 'Q' to quit", "CT"
  for i = 1 to NMENUS
    y = MENUY + (i-1)*MENUI
    box MENUX, y, MENUW, MENUH
    select case i
      case 1
        text MENUX+10, y+15, "Speed WPM (5.0 to 60.0): "
        text MENUXV, y+10, str$(wpm_speed), "LT", 3
      case 2
        msg$ = "Frequency (" + str$(MIN_FREQ) + " to " + str$(MAX_FREQ) + "Hz): "
        text MENUX+10, y+15, msg$
        text MENUXV, y+10, str$(frequency), "LT", 3
      case 3
        msg$ = "Text Batch Size (" + str$(MIN_BATCH) + " to " + str$(MAX_BATCH) + "): "
        text MENUX+10, y+15, msg$
        text MENUXV, y+10, str$(batch_size), "LT", 3
      case 4
        text MENUX+10, y+15, "Text from File: "
      case 5
        text MENUX+10, y+15, "Random Alphabetic Chars: "
      case 6
        text MENUX+10, y+15, "Random Alphabetic and Number Chars: "
      case 7
        text MENUX+10, y+15, "Random Alphabetic, Number, and Punctuation Chars: "
      case 8
        text MENUX+10, y+15, "Koch Incremental Learning Method: "
        text MENUXV, y+10, str$(num_koch_used), "LT", 3
    end select
  next i
  z$ = INKEY$
  choice = 1
  HiliteMenu choice
  do
    do
      z$ = INKEY$
    loop until z$ <> ""
    cmd = asc(z$)
    select case cmd
      case QUERY
        ShowHelpScreen
        exit sub
      case LQUERY
        ShowHelpScreen
        exit sub
      case QUIT
        close #1
        cls
        end
      case LQUIT
        close #1
        cls
        end
      case UP
        choice = choice-1
        if choice < 1 then choice = 1
        HiliteMenu choice
      case DOWN
        choice = choice+1
        if choice > NMENUS then choice = NMENUS
        HiliteMenu choice
      case LEFT
        y = MENUY + (choice-1)*MENUI
        if choice = 1 then
          wpm_speed = wpm_speed-1
          if wpm_speed <= MIN_SPEED then wpm_speed = MIN_SPEED
          text MENUXV, y+10, "     ", "LT", 3          
          text MENUXV, y+10, str$(wpm_speed), "LT", 3
        else if choice = 2 then
          frequency = frequency - 50
          if frequency <= MIN_FREQ then frequency = MIN_FREQ
          text MENUXV, y+10, "     ", "LT", 3          
          text MENUXV, y+10, str$(frequency), "LT", 3
        else if choice = 3 then
          batch_size = batch_size-10
          if batch_size <= MIN_BATCH then batch_size = MIN_BATCH
          text MENUXV, y+10, "     ", "LT", 3          
          text MENUXV, y+10, str$(batch_size), "LT", 3
        else if choice = 8 then
          num_koch_used = num_koch_used-1
          if num_koch_used < 2 then num_koch_used = 2
          text MENUXV, y+10, "     ", "LT", 3          
          text MENUXV, y+10, str$(num_koch_used), "LT", 3
        end if
      case RIGHT   
        y = MENUY + (choice-1)*MENUI
        if choice = 1 then
          wpm_speed = wpm_speed+1
          if wpm_speed >= MAX_SPEED then wpm_speed = MAX_SPEED
          text MENUXV, y+10, "     ", "LT", 3          
          text MENUXV, y+10, str$(wpm_speed), "LT", 3
        else if choice = 2 then
          frequency = frequency + 50
          if frequency >= MAX_FREQ then frequency = MAX_FREQ
          text MENUXV, y+10, "     ", "LT", 3          
          text MENUXV, y+10, str$(frequency), "LT", 3
        else if choice = 3 then
          batch_size = batch_size+10
          if batch_size >= MAX_BATCH then batch_size = MAX_BATCH
          text MENUXV, y+10, "     ", "LT", 3          
          text MENUXV, y+10, str$(batch_size), "LT", 3
        else if choice = 8 then
          num_koch_used = num_koch_used+1
          if num_koch_used > NUM_KOCH then num_koch_used = NUM_KOCH  
          text MENUXV, y+10, "     ", "LT", 3          
          text MENUXV, y+10, str$(num_koch_used), "LT", 3
        end if
      case ENTER
        exit do
    end select
  loop
  ExecuteChoice choice
end sub    

' Highlight the currently active Menu item
sub HiliteMenu which
  local x, y
  local xv(4), yv(4)
  x = MENUX-10
  if prev_choice > 0 then
    y = MENUY + (prev_choice-1)*MENUI + MENUH\2
    xv(1) = x           : yv(1) = y
    xv(2) = x - MENUH\2 : yv(2) = y - MENUH\2
    xv(3) = xv(2)       : yv(3) = y + MENUH\2
    xv(4) = xv(1)       : yv(4) = yv(1)
    polygon 4, xv(), yv(), RGB(BLACK), RGB(BLACK)
  end if
  y = MENUY + (which-1)*MENUI + MENUH\2
  xv(1) = x           : yv(1) = y
  xv(2) = x - MENUH\2 : yv(2) = y - MENUH\2
  xv(3) = xv(2)       : yv(3) = y + MENUH\2
  xv(4) = xv(1)       : yv(4) = yv(1)
  polygon 4, xv(), yv(), RGB(RED), RGB(RED)
  prev_choice = which
end sub

' Linking subroutine
sub ExecuteChoice choice
  if choice < 4 then choice = MTXT2
  SavePreferences
  select case choice
    case MTXT1
      text_mode = FROM_FILE
    case MTXT2
      text_mode = RAND_ALPHA
    case MTXT3
      text_mode = RAND_ALPHA_NUM
    case MTXT4  
      text_mode = RAND_ALPHA_NUM_PUNCT
    case MKOCH
      text_mode = KOCH_INCREMENTAL
  end select
  ShowPracticeScreen
end sub

' Show the MorseCode practice screen
sub ShowPracticeScreen
  local x, y, add_spaces
  local z$, theText$

  cls
  MakeDurations
  x = 50 : y = 50
  text MM.HRES\2, 20, "Code Practice Screen", "CT", 3
  box x, y, 749, 500
  if text_mode = KOCH_INCREMENTAL then
    text x+10, y+15, "Mode: " + modename$(text_mode) + " (" + str$(num_koch_used) + " chars)"
  else
    text x+10, y+15, "Mode: " + modename$(text_mode)
  end if
  text x+10, y+30, "Speed: " + str$(wpm_speed) + " words per minute"
  text x+300, y+30, "Frequency: " + str$(frequency)
  text x+500, y+30, "Batch size: " + str$(batch_size)
  text x+10, y+50, "Press Any Key to Start Practice"
  line x, y+75, x+749, y+75
  line x, y+500-50, x+749, y+500-50
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""  
  add_spaces = 0
  select case text_mode
    case FROM_FILE
      EmitTextCode theText$
    case RAND_ALPHA to RAND_ALPHA_NUM_PUNCT
      EmitRandCode theText$
      add_spaces = 1
    case KOCH_INCREMENTAL
      EmitKochCode theText$
  end select
  ShowText x+10, y+80, theText$, add_spaces
  y = y+500-50
  text x+10, y+15, "Press 'M' or Enter to return to menu."
  text X+10, y+30, "Press 'Q' to quit program.
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  cmd = asc(z$)
  select case cmd
    case 77
      exit sub
    case 109
      exit sub
    case 81
      close #1
      cls
      end
    case 113
      close #1
      cls
      end
    case ENTER
      exit sub
  end select
end sub

' after morse code emission for a batch is done,
' show the text from which it was generated.
sub ShowText sx, sy, theText$, add_spaces
  local nlines, npl nlast, i, p, nc, np
  local y
  local out$ = ""
  if add_spaces then
    for i = 1 to len(theText$)
      if len(out$) < 254 then      
        out$ = out$ + MID$(theText$, i, 1) + " "
      end if
    next i
  else
    out$ = theText$
  end if
  nc = 58
  nlines = (len(out$) + nc - 1)\nc
  nlast = len(out$) - (nlines-1)*nc
  p = 1
  np = nc
  y = sy+15
  for i = 1 to nlines
    if i = nlines then np = nlast
    text sx, y, MID$(out$, p, nc),, 2,, RGB(BLUE)
    p = p+nc
    y = y+30
  next i
end sub

' Emit Morse Code from randomly-chosen section of
' Travesty-generated text file
sub EmitTextCode out$
  local i, tlen, soff, sline, schar, ncpl, brlen, qlen, stop
  tlen = 0
  for i = 1 to RTEXT_NLINES
    tlen = tlen+rtext_lens(i)
  next i
  start = int(rnd()*(tlen-batch_size))
  soff = 0
  for i = 1 to RTEXT_NLINES
    soff = soff+rtext_lens(i)
    if soff >= start then
      sline = i
      schar = soff - start
      exit for
    end if
  next i
  brlen = batch_size
  out$ = ""
  for i = 1 to batch_size
    ncpl = rtext_lens(sline) - schar + 1
    if ncpl > brlen then ncpl = brlen
    stop = 0
    qlen = len(out$) + ncpl
    if qlen > 254 then 
      ncpl = qlen - 254
      stop = 1
    end if
    out$ = out$ + MID$(rtext_lines$(sline), schar, ncpl)
    if stop then exit for
    brlen = brlen - ncpl
    if brlen > 0 then
      sline = sline+1
      schar = 1
    else
      exit for
    end if
  next i
  EmitMorseString out$
end sub

' Prepare and emit a string composed of random characters,
' either pure alphabetic, or mixed alphabetic, numbers, and
' punctuation, depending on the text_mode.
sub EmitRandCode out$
  local chars$(batch_size)
  'local out$
  local i, ax, n, nx, px
  for i = 1 to batch_size
    ax = 1 + int(rnd()*NUM_ALPHA)
    chars$(i) = chr$(ALPHA_ASC_START + ax - 1)
  next i
  if text_mode = RAND_ALPHA_NUM or text_mode = RAND_ALPHA_NUM_PUNCT then
    n = int(batch_size/8)
    for i = 1 to n
      nx = 1 + int(rnd()*NUM_NUMBERS)
      ax = 1 + int(rnd()*batch_size)
      chars$(ax) = chr$(NUM_ASC_START + nx - 1)
    next i
  end if
  if text_mode = RAND_ALPHA_NUM_PUNCT then
    n = int(batch_size/8)
    for i = 1 to n
      px = 1 + int(rnd()*NUM_PUNCT)
      ax = 1 + int(rnd()*batch_size)
      chars$(ax) = chr$(punct_ascii(px))
    next i
  end if
  out$ = ""
  for i = 1 to batch_size
    out$ = out$ + chars$(i)
  next i
  EmitMorseString out$
end sub

' Emit Morse Code sounds using the Koch Incremental
' Learning System.
sub EmitKochCode msg$
  local w, n, kx
  print #1, "EmitKochCode"
  w = 0
  n = 0
  msg$ = ""
  do while n < batch_size
    kx = 1 + int(rnd()*num_koch_used)
    msg$ = msg$ + koch_order_char$(kx)
    w = w+1
    n = n+1
    if w = KOCH_WORD_LENGTH then
      msg$ = msg$ + " "
      n = n+1
      w = 0
    end if
  loop
  EmitMorseString msg$
end sub

' Emit Morse Code sounds for a String
sub EmitMorseString msg$
  local i, k
  local words$(255)
  msg$ = UCASE$(msg$)
  k = 1
  do
    words$(k) = FIELD$(msg$, k, " ")
    if len(words$(k)) > 0 then
      k = k+1
    else
      exit do
    end if
  loop
  k = k-1
  for i = 1 to k
    EmitMorseWord words$(i)
  next i
end sub

' Emit Morse Code sounds for a word
sub EmitMorseWord wd$
  local i
  for i = 1 to len(wd$)
    EmitMorseChar MID$(wd$, i, 1)
  next i
  pause word_pause
end sub

' Emit Morse Code sounds for a character
sub EmitMorseChar ch$
  local i, a, ix, n, j, s
  a = asc(ch$)
  select case a
    case ALPHA_ASC_START to ALPHA_ASC_END
      ix = a - ALPHA_ASC_START + 1
      n = ALPHA_LEN
      for j = 1 to n
        s = alpha(ix, j)
        select case s
          case DIT
            play sound 1, B, T, frequency
            pause dit_duration
            play stop
            pause elt_pause
         case DAH
            play sound 1, B, T, frequency
            pause dah_duration
            play stop
            pause elt_pause
          case else
            ' ignore trailing zeros
        end select
      next j
    case NUM_ASC_START to NUM_ASC_END
      ix = a - NUM_ASC_START + 1
      n = NUM_LEN
      for j = 1 to n
        s = numbers(ix, j)
        select case s
          case DIT
            play sound 1, B, T, frequency
            pause dit_duration
            play stop
            pause elt_pause
          case DAH
            play sound 1, B, T, frequency
            pause dah_duration
            play stop
            pause elt_pause
          case else
        end select
      next j
    case else
      for i = 1 to NUM_PUNCT
        if a = punct_ascii(i) then
          ix = i
          exit for
        end if
      next i
      n = PUNCT_LEN
      for j = 1 to n
        s = punctuation(ix, j)
        select case s
          case DIT
            play sound 1, B, T, frequency
            pause dit_duration
            play stop
            pause elt_pause
          case DAH
            play sound 1, B, T, frequency
            pause dah_duration
            play stop
            pause elt_pause
          case else
            ' ignore trailing zeros
        end select
      next j
  end select
  pause char_pause
end sub

' Save Preferences to file
sub SavePreferences
  open "./Morse.sav" for output as #2
  print #2, "Morse 1.0.0"
  print #2, str$(wpm_speed)
  print #2, str$(frequency)
  print #2, str$(text_mode)
  print #2, str$(batch_size)
  print #2, str$(num_koch_used)
  close #2
end sub

' Load Preferences from file
' If file does not yet exist, do not error.
sub LoadPreferences
  buf$ = ""
  on error skip 3
  open "./Morse.sav" for input as #2
  line input #2, buf$
  close #2
  if buf$ = "" then exit sub
  open "./Morse.sav" for input as #2
  line input #2, buf$
  line input #2, buf$
  wpm_speed = val(buf$)
  line input #2, buf$
  frequency = val(buf$)
  line input #2, buf$
  text_mode = val(buf$)
  line input #2, buf$
  batch_size = val(buf$)
  line input #2, buf$
  num_koch_used = val(buf$)
  close #2
end sub

' Show minimal help
sub ShowHelpScreen
  cls
  print "The MorseCode program helps you to learn to listen to and decode Morse Code."
  print "It does not teach how to send Morse Code but that is much easier than decoding it."
  print ""
  print "You can choose how fast the code is sent, from 5 to 30 words per minute."
  print "You can also choose the frequency of the dots and dashes, from 300 to 1500 Hz."
  print "You can choose how much text will be sent in a batch, from 10 to 255 characters."
  print ""
  print "There are several different kinds of exercises you can choose:"
  print "  Randomly-selected text from a large text file of computer-generated gobbledegook."
  print "  Random alphabetic characters."
  print "  Random alphabetic characters and numbers."
  print "  Random alphabetic, numbers, and punctuation."
  print "  Koch Incremental Learning System characters."
  print "" 
  print "The Koch Incremental Learning System is a highly effective way of learning to"
  print "decode Morse Code at useful speeds. It starts with just two characters (M and K),"
  print "and gradually builds up to the full complement of alphabetic, numeric, and"
  print "punctuation characters. You control how often new characters are added to the"
  print "exercises. Instead of producing SLOW Morse, the Koch system encourages you to"
  print "do your exercises at 15 wpm or higher. You listen to just a few different codes"
  print "until your decoding accuracy is 90 percent or hight, and then add a new character."
  print ""
  print "To use the menu, press the UP and DOWN arrows to navigate to a row. If the row"
  print "displays a value, use the LEFT and RIGHT arrow keys to change the number value."
  print "Move the selected row until you choose a practice method and press ENTER to"
  print "begin a practice session.
  print ""
  print "Press any key to return to menu..."
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
end sub

' alphabetic characters A-Z
data 1,2,0,0
data 2,1,1,1
data 2,1,2,1
data 2,1,1,0
data 1,0,0,0
data 1,1,2,1
data 2,2,1,0
data 1,1,1,1
data 1,1,0,0
data 1,2,2,2
data 2,1,2,0
data 1,2,1,1
data 2,2,0,0
data 2,1,0,0
data 2,2,2,0
data 1,2,2,1
data 2,2,1,2
data 1,2,1,0
data 1,1,1,0
data 2,0,0,0
data 1,1,2,0
data 1,1,1,2
data 1,2,2,0
data 2,1,1,2
data 2,1,2,2
data 2,2,1,1

' numbers 1-9,0
data 1,2,2,2,2
data 1,1,2,2,2
data 1,1,1,2,2
data 1,1,1,1,2
data 1,1,1,1,1
data 2,1,1,1,1
data 2,2,1,1,1
data 2,2,2,1,1
data 2,2,2,2,1
data 2,2,2,2,2

' punctuation and symbols
data 1,2,1,2,1,2  ' period
data 2,2,1,1,2,2  ' comma
data 1,1,2,2,1,1  ' question
data 2,1,2,1,2,2  ' exclamation
data 1,2,2,2,2,1  ' apostrophe
data 1,2,1,1,2,1  ' quote
data 2,1,2,2,1,0  ' left paren
data 2,1,2,2,1,2  ' right paren
data 1,2,1,1,1,0  ' ampersand
data 2,2,2,1,1,1  ' colon
data 2,1,2,1,2,1  ' semicolon
data 2,1,1,2,1,0  ' fwd slash
data 1,1,2,2,1,2  ' underscore
data 2,1,1,1,2,0  ' equals
data 1,2,1,2,1,0  ' plus
data 2,1,1,1,1,2  ' minus
data 1,2,2,1,2,1  ' @ sign
data 1,2,1,2,1,0  ' AR (using '%')
data 1,1,1,1,2,1  ' SK (using '#')

' ASCII values for punctuations
data 46, 44, 63, 33, 39, 34, 40, 41, 38, 58, 59
data 47, 95, 61, 43, 45, 64, 35, 37

' Koch Training Character Inclusion Order
' Note: "%" stands for SK digram, "#" stands for AR digram
data "K", "M", "R", "S", "U", "A", "P", "T", "L", "O", "W"
data "I", ".", "N", "J", "E", "F", "0", "Y", ",", "V", "G"
data "5", "/", "Q", "9", "Z", "H", "3", "8", "B", "?", "4"
data "2", "7", "C", "1", "D", "6", "X", "=", "%", "#"

' Text Mode Names
data "Text from File", "Random Alphbetic", "Random Alphabetic and Numbers"
data "Random Alphabetic, Numbers, and Punctuation", "Koch Incremental Training"
